"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "default", {
    enumerable: true,
    get: function() {
        return StreamingMetaApiConnection;
    }
});
const _terminalState = /*#__PURE__*/ _interop_require_default(require("./terminalState"));
const _memoryHistoryStorage = /*#__PURE__*/ _interop_require_default(require("./memoryHistoryStorage"));
const _timeoutError = /*#__PURE__*/ _interop_require_default(require("../clients/timeoutError"));
const _randomstring = /*#__PURE__*/ _interop_require_default(require("randomstring"));
const _connectionHealthMonitor = /*#__PURE__*/ _interop_require_default(require("./connectionHealthMonitor"));
const _errorHandler = require("../clients/errorHandler");
const _optionsValidator = /*#__PURE__*/ _interop_require_default(require("../clients/optionsValidator"));
const _logger = /*#__PURE__*/ _interop_require_default(require("../logger"));
const _metaApiConnection = /*#__PURE__*/ _interop_require_default(require("./metaApiConnection"));
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let StreamingMetaApiConnection = class StreamingMetaApiConnection extends _metaApiConnection.default {
    /**
   * Opens the connection. Can only be called the first time, next calls will be ignored.
   * @param {string} instanceId connection instance id
   * @return {Promise} promise resolving when the connection is opened
   */ async connect(instanceId) {
        if (!this._openedInstances.includes(instanceId)) {
            this._openedInstances.push(instanceId);
        }
        if (!this._opened) {
            this._logger.trace(`${this._account.id}: Opening connection`);
            this._opened = true;
            try {
                await this.initialize();
                await this.subscribe();
            } catch (err) {
                await this.close();
                throw err;
            }
        }
    }
    /**
   * Clears the order and transaction history of a specified application and removes application
   * @return {Promise} promise resolving when the history is cleared and application is removed
   */ removeApplication() {
        this._checkIsConnectionActive();
        this._historyStorage.clear();
        return this._websocketClient.removeApplication(this._account.id);
    }
    /**
   * Requests the terminal to start synchronization process
   * (see https://metaapi.cloud/docs/client/websocket/synchronizing/synchronize/)
   * @param {String} instanceIndex instance index
   * @returns {Promise} promise which resolves when synchronization started
   */ async synchronize(instanceIndex) {
        this._checkIsConnectionActive();
        const region = this.getRegion(instanceIndex);
        const instance = this.getInstanceNumber(instanceIndex);
        const host = this.getHostName(instanceIndex);
        let startingHistoryOrderTime = new Date(Math.max((this._historyStartTime || new Date(0)).getTime(), (await this._historyStorage.lastHistoryOrderTime(instance)).getTime()));
        let startingDealTime = new Date(Math.max((this._historyStartTime || new Date(0)).getTime(), (await this._historyStorage.lastDealTime(instance)).getTime()));
        let synchronizationId = _randomstring.default.generate(32);
        this._getState(instanceIndex).lastSynchronizationId = synchronizationId;
        const accountId = this._account.accountRegions[region];
        this._logger.debug(`${this._account.id}:${instanceIndex}: initiating synchronization ${synchronizationId}`);
        return this._websocketClient.synchronize(accountId, instance, host, synchronizationId, startingHistoryOrderTime, startingDealTime, this.terminalState.getHashes());
    }
    /**
   * Initializes meta api connection
   * @return {Promise} promise which resolves when meta api connection is initialized
   */ async initialize() {
        this._checkIsConnectionActive();
        await this._historyStorage.initialize(this._account.id, this._connectionRegistry.application);
        this._websocketClient.addAccountCache(this._account.id, this._account.accountRegions);
    }
    /**
   * Initiates subscription to MetaTrader terminal
   * @returns {Promise} promise which resolves when subscription is initiated
   */ async subscribe() {
        this._checkIsConnectionActive();
        const accountRegions = this._account.accountRegions;
        Object.entries(accountRegions).forEach(([region, replicaId])=>{
            if (!this._options.region || this._options.region === region) {
                this._websocketClient.ensureSubscribe(replicaId, 0);
                this._websocketClient.ensureSubscribe(replicaId, 1);
            }
        });
    }
    /**
   * Subscribes on market data of specified symbol (see
   * https://metaapi.cloud/docs/client/websocket/marketDataStreaming/subscribeToMarketData/).
   * @param {String} symbol symbol (e.g. currency pair or an index)
   * @param {Array<MarketDataSubscription>} subscriptions array of market data subscription to create or update. Please
   * note that this feature is not fully implemented on server-side yet
   * @param {number} [timeoutInSeconds] timeout to wait for prices in seconds, default is 30
   * @param {boolean} [waitForQuote] if set to false, the method will resolve without waiting for the first quote to
   * arrive. Default is to wait for quote if quotes subscription is requested.
   * @returns {Promise} promise which resolves when subscription request was processed
   */ async subscribeToMarketData(symbol, subscriptions, timeoutInSeconds, waitForQuote = true) {
        this._checkIsConnectionActive();
        if (!this._terminalState.specification(symbol)) {
            throw new _errorHandler.ValidationError(`${this._account.id}: Cannot subscribe to market data for symbol ${symbol} because ` + "symbol does not exist");
        } else {
            subscriptions = subscriptions || [
                {
                    type: "quotes"
                }
            ];
            if (this._subscriptions[symbol]) {
                const prevSubscriptions = this._subscriptions[symbol].subscriptions;
                subscriptions.forEach((subscription)=>{
                    const index = subscription.type === "candles" ? prevSubscriptions.findIndex((item)=>item.type === subscription.type && item.timeframe === subscription.timeframe) : prevSubscriptions.findIndex((item)=>item.type === subscription.type);
                    if (index === -1) {
                        prevSubscriptions.push(subscription);
                    } else {
                        prevSubscriptions[index] = subscription;
                    }
                });
            } else {
                this._subscriptions[symbol] = {
                    subscriptions
                };
            }
            await this._websocketClient.subscribeToMarketData(this._account.id, symbol, subscriptions, this._account.reliability);
            if (waitForQuote !== false && subscriptions.find((s)=>s.type === "quotes")) {
                return this.terminalState.waitForPrice(symbol, timeoutInSeconds);
            }
        }
    }
    /**
   * Unsubscribes from market data of specified symbol (see
   * https://metaapi.cloud/docs/client/websocket/marketDataStreaming/unsubscribeFromMarketData/).
   * @param {String} symbol symbol (e.g. currency pair or an index)
   * @param {Array<MarketDataUnsubscription>} unsubscriptions array of subscriptions to cancel
   * @returns {Promise} promise which resolves when unsubscription request was processed
   */ unsubscribeFromMarketData(symbol, unsubscriptions) {
        this._checkIsConnectionActive();
        if (!unsubscriptions) {
            delete this._subscriptions[symbol];
        } else if (this._subscriptions[symbol]) {
            this._subscriptions[symbol].subscriptions = this._subscriptions[symbol].subscriptions.filter((subscription)=>{
                return !unsubscriptions.find((unsubscription)=>subscription.type === unsubscription.type && (!unsubscription.timeframe || subscription.timeframe === unsubscription.timeframe));
            });
            if (!this._subscriptions[symbol].subscriptions.length) {
                delete this._subscriptions[symbol];
            }
        }
        return this._websocketClient.unsubscribeFromMarketData(this._account.id, symbol, unsubscriptions, this._account.reliability);
    }
    /**
   * Invoked when subscription downgrade has occurred
   * @param {String} instanceIndex index of an account instance connected
   * @param {string} symbol symbol to update subscriptions for
   * @param {Array<MarketDataSubscription>} updates array of market data subscription to update
   * @param {Array<MarketDataUnsubscription>} unsubscriptions array of subscriptions to cancel
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ // eslint-disable-next-line complexity
    async onSubscriptionDowngraded(instanceIndex, symbol, updates, unsubscriptions) {
        if (unsubscriptions === null || unsubscriptions === void 0 ? void 0 : unsubscriptions.length) {
            this.unsubscribeFromMarketData(symbol, unsubscriptions).catch((err)=>{
                let method = err.name !== "ValidationError" ? "error" : "trace";
                this._logger[method](`${this._account.id}: failed do unsubscribe from market data on subscription downgraded`, err);
            });
        }
        if (updates === null || updates === void 0 ? void 0 : updates.length) {
            this.subscribeToMarketData(symbol, updates).catch((err)=>{
                this._logger.error(`${this._account.id}: failed do subscribe from market data on subscription downgraded`, err);
            });
        }
    }
    /**
   * Returns list of the symbols connection is subscribed to
   * @returns {Array<String>} list of the symbols connection is subscribed to
   */ get subscribedSymbols() {
        return Object.keys(this._subscriptions);
    }
    /**
   * Returns subscriptions for a symbol
   * @param {string} symbol symbol to retrieve subscriptions for
   * @returns {Array<MarketDataSubscription>} list of market data subscriptions for the symbol
   */ subscriptions(symbol) {
        this._checkIsConnectionActive();
        return (this._subscriptions[symbol] || {}).subscriptions;
    }
    /**
   * Returns local copy of terminal state
   * @returns {TerminalState} local copy of terminal state
   */ get terminalState() {
        return this._terminalState;
    }
    /**
   * Returns local history storage
   * @returns {HistoryStorage} local history storage
   */ get historyStorage() {
        return this._historyStorage;
    }
    /**
   * Invoked when connection to MetaTrader terminal established
   * @param {String} instanceIndex index of an account instance connected
   * @param {Number} replicas number of account replicas launched
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ async onConnected(instanceIndex, replicas) {
        let key = _randomstring.default.generate(32);
        let state = this._getState(instanceIndex);
        const region = this.getRegion(instanceIndex);
        this.cancelRefresh(region);
        await this._terminalHashManager.refreshIgnoredFieldLists(region);
        state.shouldSynchronize = key;
        state.synchronizationRetryIntervalInSeconds = 1;
        state.synchronized = false;
        this._ensureSynchronized(instanceIndex, key);
        this._logger.debug(`${this._account.id}:${instanceIndex}: connected to broker`);
    }
    /**
   * Invoked when connection to MetaTrader terminal terminated
   * @param {String} instanceIndex index of an account instance connected
   */ async onDisconnected(instanceIndex) {
        let state = this._getState(instanceIndex);
        state.lastDisconnectedSynchronizationId = state.lastSynchronizationId;
        state.lastSynchronizationId = undefined;
        state.shouldSynchronize = undefined;
        state.synchronized = false;
        state.disconnected = true;
        const instanceNumber = this.getInstanceNumber(instanceIndex);
        const region = this.getRegion(instanceIndex);
        const instance = `${region}:${instanceNumber}`;
        delete this._refreshMarketDataSubscriptionSessions[instance];
        clearTimeout(this._refreshMarketDataSubscriptionTimeouts[instance]);
        delete this._refreshMarketDataSubscriptionTimeouts[instance];
        clearTimeout(state.synchronizationTimeout);
        delete state.synchronizationTimeout;
        clearTimeout(state.ensureSynchronizeTimeout);
        delete state.ensureSynchronizeTimeout;
        this._logger.debug(`${this._account.id}:${instanceIndex}: disconnected from broker`);
    }
    /**
   * Invoked when a symbol specifications were updated
   * @param {String} instanceIndex index of account instance connected
   * @param {Array<MetatraderSymbolSpecification>} specifications updated specifications
   * @param {Array<String>} removedSymbols removed symbols
   */ onSymbolSpecificationsUpdated(instanceIndex, specifications, removedSymbols) {
        this._scheduleSynchronizationTimeout(instanceIndex);
    }
    /**
   * Invoked when position synchronization finished to indicate progress of an initial terminal state synchronization
   * @param {string} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   */ onPositionsSynchronized(instanceIndex, synchronizationId) {
        this._scheduleSynchronizationTimeout(instanceIndex);
    }
    /**
   * Invoked when pending order synchronization fnished to indicate progress of an initial terminal state
   * synchronization
   * @param {string} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   */ onPendingOrdersSynchronized(instanceIndex, synchronizationId) {
        this._scheduleSynchronizationTimeout(instanceIndex);
    }
    /**
   * Invoked when a synchronization of history deals on a MetaTrader account have finished to indicate progress of an
   * initial terminal state synchronization
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ async onDealsSynchronized(instanceIndex, synchronizationId) {
        let state = this._getState(instanceIndex);
        state.dealsSynchronized[synchronizationId] = true;
        this._scheduleSynchronizationTimeout(instanceIndex);
        this._logger.debug(`${this._account.id}:${instanceIndex}: finished synchronization ${synchronizationId}`);
    }
    /**
   * Invoked when a synchronization of history orders on a MetaTrader account have finished to indicate progress of an
   * initial terminal state synchronization
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ async onHistoryOrdersSynchronized(instanceIndex, synchronizationId) {
        let state = this._getState(instanceIndex);
        state.ordersSynchronized[synchronizationId] = true;
        this._scheduleSynchronizationTimeout(instanceIndex);
    }
    /**
   * Invoked when connection to MetaApi websocket API restored after a disconnect
   * @param {String} region reconnected region
   * @param {Number} instanceNumber reconnected instance number
   * @return {Promise} promise which resolves when connection to MetaApi websocket API restored after a disconnect
   */ async onReconnected(region, instanceNumber) {
        const instanceTemplate = `${region}:${instanceNumber}`;
        Object.keys(this._stateByInstanceIndex).filter((key)=>key.startsWith(`${instanceTemplate}:`)).forEach((key)=>{
            delete this._stateByInstanceIndex[key];
        });
        delete this._refreshMarketDataSubscriptionSessions[instanceTemplate];
        clearTimeout(this._refreshMarketDataSubscriptionTimeouts[instanceTemplate]);
        delete this._refreshMarketDataSubscriptionTimeouts[instanceTemplate];
    }
    /**
   * Invoked when a stream for an instance index is closed
   * @param {String} instanceIndex index of an account instance connected
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ async onStreamClosed(instanceIndex) {
        delete this._stateByInstanceIndex[instanceIndex];
    }
    /**
   * Invoked when MetaTrader terminal state synchronization is started
   * @param {string} instanceIndex index of an account instance connected
   * @param {string} specificationsHash specifications hash
   * @param {string} positionsHash positions hash
   * @param {string} ordersHash orders hash
   * @param {string} synchronizationId synchronization id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ async onSynchronizationStarted(instanceIndex, specificationsHash, positionsHash, ordersHash, synchronizationId) {
        this._logger.debug(`${this._account.id}:${instanceIndex}: starting synchronization ${synchronizationId}`);
        const instanceNumber = this.getInstanceNumber(instanceIndex);
        const region = this.getRegion(instanceIndex);
        const instance = `${region}:${instanceNumber}`;
        const accountId = this._account.accountRegions[region];
        delete this._refreshMarketDataSubscriptionSessions[instance];
        let sessionId = _randomstring.default.generate(32);
        this._refreshMarketDataSubscriptionSessions[instance] = sessionId;
        clearTimeout(this._refreshMarketDataSubscriptionTimeouts[instance]);
        delete this._refreshMarketDataSubscriptionTimeouts[instance];
        await this._refreshMarketDataSubscriptions(accountId, instanceNumber, sessionId);
        this._scheduleSynchronizationTimeout(instanceIndex);
        let state = this._getState(instanceIndex);
        if (state && !this._closed) {
            state.lastSynchronizationId = synchronizationId;
        }
    }
    /**
   * Invoked when account region has been unsubscribed
   * @param {String} region account region unsubscribed
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ onUnsubscribeRegion(region) {
        Object.keys(this._refreshMarketDataSubscriptionTimeouts).filter((instance)=>instance.startsWith(`${region}:`)).forEach((instance)=>{
            clearTimeout(this._refreshMarketDataSubscriptionTimeouts[instance]);
            delete this._refreshMarketDataSubscriptionTimeouts[instance];
            delete this._refreshMarketDataSubscriptionSessions[instance];
        });
        Object.keys(this._stateByInstanceIndex).filter((instance)=>instance.startsWith(`${region}:`)).forEach((instance)=>delete this._stateByInstanceIndex[instance]);
    }
    /**
   * Returns flag indicating status of state synchronization with MetaTrader terminal
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} synchronizationId optional synchronization request id, last synchronization request id will be used
   * by default
   * @return {Promise<Boolean>} promise resolving with a flag indicating status of state synchronization with MetaTrader
   * terminal
   */ async isSynchronized(instanceIndex, synchronizationId) {
        return Object.values(this._stateByInstanceIndex).reduce((acc, s)=>{
            if (instanceIndex !== undefined && s.instanceIndex !== instanceIndex) {
                return acc;
            }
            const checkSynchronizationId = synchronizationId || s.lastSynchronizationId;
            let synchronized = !!s.ordersSynchronized[checkSynchronizationId] && !!s.dealsSynchronized[checkSynchronizationId];
            return acc || synchronized;
        }, false);
    }
    /**
   * @typedef {Object} SynchronizationOptions
   * @property {String} [applicationPattern] application regular expression pattern, default is .*
   * @property {String} [synchronizationId] synchronization id, last synchronization request id will be used by
   * default
   * @property {Number} [instanceIndex] index of an account instance to ensure synchronization on, default is to wait
   * for the first instance to synchronize
   * @property {Number} [timeoutInSeconds] wait timeout in seconds, default is 5m
   * @property {Number} [intervalInMilliseconds] interval between account reloads while waiting for a change, default is 1s
   */ /**
   * Waits until synchronization to MetaTrader terminal is completed
   * @param {SynchronizationOptions} opts synchronization options
   * @return {Promise} promise which resolves when synchronization to MetaTrader terminal is completed
   * @throws {TimeoutError} if application failed to synchronize with the teminal within timeout allowed
   */ // eslint-disable-next-line complexity
    async waitSynchronized(opts) {
        this._checkIsConnectionActive();
        opts = opts || {};
        let instanceIndex = opts.instanceIndex;
        let synchronizationId = opts.synchronizationId;
        let timeoutInSeconds = opts.timeoutInSeconds || 300;
        let intervalInMilliseconds = opts.intervalInMilliseconds || 1000;
        let applicationPattern = opts.applicationPattern || (this._account.application === "CopyFactory" ? "CopyFactory.*|RPC" : "RPC");
        let startTime = Date.now();
        let synchronized;
        while(!(synchronized = await this.isSynchronized(instanceIndex, synchronizationId)) && startTime + timeoutInSeconds * 1000 > Date.now()){
            await new Promise((res)=>setTimeout(res, intervalInMilliseconds));
        }
        let state;
        if (instanceIndex === undefined) {
            for (let s of Object.values(this._stateByInstanceIndex)){
                if (await this.isSynchronized(s.instanceIndex, synchronizationId)) {
                    state = s;
                    instanceIndex = s.instanceIndex;
                }
            }
        } else {
            state = Object.values(this._stateByInstanceIndex).find((s)=>s.instanceIndex === instanceIndex);
        }
        if (!synchronized) {
            throw new _timeoutError.default("Timed out waiting for MetaApi to synchronize to MetaTrader account " + this._account.id + ", synchronization id " + (synchronizationId || state && state.lastSynchronizationId || state && state.lastDisconnectedSynchronizationId));
        }
        let timeLeftInSeconds = Math.max(0, timeoutInSeconds - (Date.now() - startTime) / 1000);
        const region = this.getRegion(state.instanceIndex);
        const accountId = this._account.accountRegions[region];
        await this._websocketClient.waitSynchronized(accountId, this.getInstanceNumber(instanceIndex), applicationPattern, timeLeftInSeconds);
    }
    /**
   * Closes the connection. The instance of the class should no longer be used after this method is invoked.
   * @param {string} instanceId connection instance id
   */ async close(instanceId) {
        if (this._opened) {
            this._openedInstances = this._openedInstances.filter((id)=>id !== instanceId);
            if (!this._openedInstances.length && !this._closed) {
                clearInterval(this._refreshJob);
                this._logger.debug(`${this._account.id}: Closing connection`);
                Object.values(this._stateByInstanceIndex).forEach((state)=>clearTimeout(state.synchronizationTimeout));
                this._stateByInstanceIndex = {};
                await this._connectionRegistry.removeStreaming(this._account);
                this._terminalState.close();
                const accountRegions = this._account.accountRegions;
                this._websocketClient.removeSynchronizationListener(this._account.id, this);
                this._websocketClient.removeSynchronizationListener(this._account.id, this._terminalState);
                this._websocketClient.removeSynchronizationListener(this._account.id, this._historyStorage);
                this._websocketClient.removeSynchronizationListener(this._account.id, this._healthMonitor);
                this._websocketClient.removeReconnectListener(this);
                this._healthMonitor.stop();
                this._refreshMarketDataSubscriptionSessions = {};
                Object.values(this._refreshMarketDataSubscriptionTimeouts).forEach((timeout)=>clearTimeout(timeout));
                this._refreshMarketDataSubscriptionTimeouts = {};
                Object.values(accountRegions).forEach((replicaId)=>this._websocketClient.removeAccountCache(replicaId));
                this._closed = true;
                this._logger.trace(`${this._account.id}: Closed connection`);
            }
        }
    }
    /**
   * Returns synchronization status
   * @return {boolean} synchronization status
   */ get synchronized() {
        return Object.values(this._stateByInstanceIndex).reduce((acc, s)=>acc || s.synchronized, false);
    }
    /**
   * Returns MetaApi account
   * @return {MetatraderAccount} MetaApi account
   */ get account() {
        return this._account;
    }
    /**
   * Returns connection health monitor instance
   * @return {ConnectionHealthMonitor} connection health monitor instance
   */ get healthMonitor() {
        return this._healthMonitor;
    }
    async _refreshMarketDataSubscriptions(accountId, instanceNumber, session) {
        const region = this._websocketClient.getAccountRegion(accountId);
        const instance = `${region}:${instanceNumber}`;
        try {
            if (this._refreshMarketDataSubscriptionSessions[instance] === session) {
                const subscriptionsList = [];
                Object.keys(this._subscriptions).forEach((key)=>{
                    const subscriptions = this.subscriptions(key);
                    const subscriptionsItem = {
                        symbol: key
                    };
                    if (subscriptions) {
                        subscriptionsItem.subscriptions = subscriptions;
                    }
                    subscriptionsList.push(subscriptionsItem);
                });
                await this._websocketClient.refreshMarketDataSubscriptions(accountId, instanceNumber, subscriptionsList);
            }
        } catch (err) {
            this._logger.error(`Error refreshing market data subscriptions job for account ${this._account.id} ` + `${instanceNumber}`, err);
        } finally{
            if (this._refreshMarketDataSubscriptionSessions[instance] === session) {
                let refreshInterval = (Math.random() * (this._maxSubscriptionRefreshInterval - this._minSubscriptionRefreshInterval) + this._minSubscriptionRefreshInterval) * 1000;
                this._refreshMarketDataSubscriptionTimeouts[instance] = setTimeout(()=>this._refreshMarketDataSubscriptions(accountId, instanceNumber, session), refreshInterval);
            }
        }
    }
    _generateStopOptions(stopLoss, takeProfit) {
        let trade = {};
        if (typeof stopLoss === "number") {
            trade.stopLoss = stopLoss;
        } else if (stopLoss) {
            trade.stopLoss = stopLoss.value;
            trade.stopLossUnits = stopLoss.units;
        }
        if (typeof takeProfit === "number") {
            trade.takeProfit = takeProfit;
        } else if (takeProfit) {
            trade.takeProfit = takeProfit.value;
            trade.takeProfitUnits = takeProfit.units;
        }
        return trade;
    }
    async _ensureSynchronized(instanceIndex, key) {
        let state = this._getState(instanceIndex);
        if (state && state.shouldSynchronize && !this._closed) {
            try {
                const synchronizationResult = await this.synchronize(instanceIndex);
                if (synchronizationResult) {
                    state.synchronized = true;
                    state.synchronizationRetryIntervalInSeconds = 1;
                    delete state.ensureSynchronizeTimeout;
                }
                this._scheduleSynchronizationTimeout(instanceIndex);
            } catch (err) {
                const level = this._latencyService.getSynchronizedAccountInstances(this._account.id).length ? "debug" : "error";
                this._logger[level]("MetaApi websocket client for account " + this._account.id + ":" + instanceIndex + " failed to synchronize", err);
                if (state.shouldSynchronize === key) {
                    clearTimeout(state.ensureSynchronizeTimeout);
                    state.ensureSynchronizeTimeout = setTimeout(this._ensureSynchronized.bind(this, instanceIndex, key), state.synchronizationRetryIntervalInSeconds * 1000);
                    state.synchronizationRetryIntervalInSeconds = Math.min(state.synchronizationRetryIntervalInSeconds * 2, 300);
                }
            }
        }
    }
    _getState(instanceIndex) {
        if (!this._stateByInstanceIndex["" + instanceIndex]) {
            this._stateByInstanceIndex["" + instanceIndex] = {
                instanceIndex,
                ordersSynchronized: {},
                dealsSynchronized: {},
                shouldSynchronize: undefined,
                synchronizationRetryIntervalInSeconds: 1,
                synchronized: false,
                lastDisconnectedSynchronizationId: undefined,
                lastSynchronizationId: undefined,
                disconnected: false
            };
        }
        return this._stateByInstanceIndex["" + instanceIndex];
    }
    _scheduleSynchronizationTimeout(instanceIndex) {
        let state = this._getState(instanceIndex);
        if (state && !this._closed) {
            clearTimeout(state.synchronizationTimeout);
            state.synchronizationTimeout = setTimeout(()=>this._checkSynchronizationTimedOut(instanceIndex), 2 * 60 * 1000);
            this._logger.debug(`${this._account.id}:${instanceIndex}: scheduled synchronization timeout`);
        }
    }
    _checkSynchronizationTimedOut(instanceIndex) {
        this._logger.debug(`${this._account.id}:${instanceIndex}: checking if synchronization timed out out`);
        let state = this._getState(instanceIndex);
        if (state && !this._closed) {
            let synchronizationId = state.lastSynchronizationId;
            let synchronized = !!state.dealsSynchronized[synchronizationId];
            if (!synchronized && synchronizationId && state.shouldSynchronize) {
                this._logger.warn(`${this._account.id}:${instanceIndex}: resynchronized since latest synchronization ` + `${synchronizationId} did not finish in time`);
                this._ensureSynchronized(instanceIndex, state.shouldSynchronize);
            }
        }
    }
    /**
   * Constructs MetaApi MetaTrader streaming Api connection
   * @param {MetaApiOpts} options metaapi options
   * @param {MetaApiWebsocketClient} websocketClient MetaApi websocket client
   * @param {TerminalHashManager} terminalHashManager terminal hash manager
   * @param {MetatraderAccount} account MetaTrader account id to connect to
   * @param {HistoryStorage} historyStorage terminal history storage. By default an instance of MemoryHistoryStorage
   * will be used.
   * @param {ConnectionRegistry} connectionRegistry metatrader account connection registry
   * @param {Date} [historyStartTime] history start sync time
   * @param {RefreshSubscriptionsOpts} [refreshSubscriptionsOpts] subscriptions refresh options
   */ constructor(options, websocketClient, terminalHashManager, account, historyStorage, connectionRegistry, historyStartTime, refreshSubscriptionsOpts){
        super(options, websocketClient, account);
        refreshSubscriptionsOpts = refreshSubscriptionsOpts || {};
        const validator = new _optionsValidator.default();
        this._minSubscriptionRefreshInterval = validator.validateNonZero(refreshSubscriptionsOpts.minDelayInSeconds, 1, "refreshSubscriptionsOpts.minDelayInSeconds");
        this._maxSubscriptionRefreshInterval = validator.validateNonZero(refreshSubscriptionsOpts.maxDelayInSeconds, 600, "refreshSubscriptionsOpts.maxDelayInSeconds");
        this._connectionRegistry = connectionRegistry;
        this._historyStartTime = historyStartTime;
        this._terminalHashManager = terminalHashManager;
        this._terminalState = new _terminalState.default(account, terminalHashManager, this._websocketClient);
        this._historyStorage = historyStorage || new _memoryHistoryStorage.default();
        this._healthMonitor = new _connectionHealthMonitor.default(this);
        this._websocketClient.addSynchronizationListener(account.id, this);
        this._websocketClient.addSynchronizationListener(account.id, this._terminalState);
        this._websocketClient.addSynchronizationListener(account.id, this._historyStorage);
        this._websocketClient.addSynchronizationListener(account.id, this._healthMonitor);
        Object.values(account.accountRegions).forEach((replicaId)=>this._websocketClient.addReconnectListener(this, replicaId));
        this._subscriptions = {};
        this._stateByInstanceIndex = {};
        this._refreshMarketDataSubscriptionSessions = {};
        this._refreshMarketDataSubscriptionTimeouts = {};
        this._openedInstances = [];
        this._logger = _logger.default.getLogger("MetaApiConnection");
    }
};

//# sourceMappingURL=data:application/json;base64,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