'use strict';
function asyncGeneratorStep(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
import randomstring from 'randomstring';
import SynchronizationListener from '../../../clients/metaApi/synchronizationListener';
import LoggerManager from '../../../logger';
let EquityChartStreamManager = class EquityChartStreamManager {
    /**
   * Returns listeners for account
   * @param {String} accountId account id to return listeners for
   * @returns {{[listenerId: string]: EquityChartListener}} dictionary of account equity chart event listeners
   */ getAccountListeners(accountId) {
        if (!this._equityChartListeners[accountId]) {
            this._equityChartListeners[accountId] = {};
        }
        return this._equityChartListeners[accountId];
    }
    /**
   * Adds an equity chart event listener
   * @param {EquityChartListener} listener equity chart event listener
   * @param {String} accountId account id
   * @param {Date} [startTime] date to start tracking from
   * @returns {Promise<string>} listener id
   */ // eslint-disable-next-line max-statements, complexity
    addEquityChartListener(listener, accountId, startTime) {
        var _this = this;
        return _async_to_generator(function*() {
            if (!_this._equityChartCaches[accountId]) {
                _this._equityChartCaches[accountId] = {
                    record: {},
                    lastPeriod: {},
                    pendingInitalizationResolves: []
                };
            }
            const cache = _this._equityChartCaches[accountId];
            let connection = null;
            let retryIntervalInSeconds = _this._retryIntervalInSeconds;
            const equityTrackingClient = _this._equityTrackingClient;
            const getAccountListeners = ()=>_this.getAccountListeners(accountId);
            const pendingInitalizationResolves = _this._pendingInitalizationResolves;
            const synchronizationFlags = _this._accountSynchronizationFlags;
            let EquityChartStreamListener = class EquityChartStreamListener extends SynchronizationListener {
                onDealsSynchronized(instanceIndex, synchronizationId) {
                    var _this = this;
                    return _async_to_generator(function*() {
                        try {
                            if (!synchronizationFlags[accountId]) {
                                synchronizationFlags[accountId] = true;
                                Object.values(getAccountListeners()).forEach((accountListener)=>{
                                    accountListener.onConnected();
                                });
                            }
                            if (pendingInitalizationResolves[accountId]) {
                                pendingInitalizationResolves[accountId].forEach((resolve)=>resolve());
                                delete pendingInitalizationResolves[accountId];
                            }
                        } catch (err) {
                            Object.values(getAccountListeners()).forEach((accountListener)=>{
                                accountListener.onError(err);
                            });
                            _this._logger.error('Error processing onDealsSynchronized event for ' + `equity chart listener for account ${accountId}`, err);
                        }
                    })();
                }
                onDisconnected(instanceIndex) {
                    var _this = this;
                    return _async_to_generator(function*() {
                        try {
                            if (synchronizationFlags[accountId] && !connection.healthMonitor.healthStatus.synchronized) {
                                synchronizationFlags[accountId] = false;
                                Object.values(getAccountListeners()).forEach((accountListener)=>{
                                    accountListener.onDisconnected();
                                });
                            }
                        } catch (err) {
                            Object.values(getAccountListeners()).forEach((accountListener)=>{
                                accountListener.onError(err);
                            });
                            _this._logger.error('Error processing onDisconnected event for ' + `equity chart listener for account ${accountId}`, err);
                        }
                    })();
                }
                // eslint-disable-next-line complexity, max-statements
                onSymbolPriceUpdated(instanceIndex, price) {
                    var _this = this;
                    return _async_to_generator(function*() {
                        try {
                            if (pendingInitalizationResolves[accountId]) {
                                pendingInitalizationResolves[accountId].forEach((resolve)=>resolve());
                                delete pendingInitalizationResolves[accountId];
                            }
                            const equity = price.equity;
                            const brokerTime = price.brokerTime;
                            if (!cache.lastPeriod) {
                                return;
                            }
                            if (brokerTime > cache.lastPeriod.endBrokerTime) {
                                Object.values(getAccountListeners()).forEach((accountListener)=>{
                                    accountListener.onEquityRecordCompleted();
                                });
                                const startBrokerTime = cache.lastPeriod.startBrokerTime;
                                cache.lastPeriod = null;
                                // eslint-disable-next-line no-constant-condition
                                while(true){
                                    let periods = yield equityTrackingClient.getEquityChart(accountId, startBrokerTime, undefined, true);
                                    if (periods.length < 2) {
                                        yield new Promise((res)=>setTimeout(res, 10000));
                                    } else {
                                        Object.values(getAccountListeners()).forEach((accountListener)=>{
                                            accountListener.onEquityRecordUpdated(periods);
                                        });
                                        cache.lastPeriod = periods[1];
                                        break;
                                    }
                                }
                            } else {
                                const accountInformation = connection.terminalState.accountInformation;
                                if (accountInformation) {
                                    const previousInfo = {
                                        startBrokerTime: cache.lastPeriod.startBrokerTime,
                                        endBrokerTime: cache.lastPeriod.endBrokerTime,
                                        averageBalance: cache.record.averageBalance,
                                        minBalance: cache.record.minBalance,
                                        maxBalance: cache.record.maxBalance,
                                        averageEquity: Math.floor(cache.record.averageEquity),
                                        minEquity: cache.record.minEquity,
                                        maxEquity: cache.record.maxEquity,
                                        lastBalance: cache.lastPeriod.lastBalance,
                                        lastEquity: cache.lastPeriod.lastEquity
                                    };
                                    let durationIncrement = new Date(brokerTime).getTime() - new Date(cache.lastPeriod.brokerTime).getTime();
                                    cache.lastPeriod.equitySum += durationIncrement * (cache.lastPeriod.equity || accountInformation.equity);
                                    cache.lastPeriod.balanceSum += durationIncrement * (cache.lastPeriod.balance || accountInformation.balance);
                                    cache.lastPeriod.duration += durationIncrement;
                                    cache.lastPeriod.equity = price.equity;
                                    cache.lastPeriod.balance = accountInformation.balance;
                                    cache.lastPeriod.brokerTime = price.brokerTime;
                                    cache.record.duration = cache.lastPeriod.duration;
                                    cache.record.balanceSum = cache.lastPeriod.balanceSum;
                                    cache.record.equitySum = cache.lastPeriod.equitySum;
                                    cache.record.averageEquity = cache.lastPeriod.duration ? cache.lastPeriod.equitySum / cache.lastPeriod.duration : equity;
                                    cache.record.averageBalance = cache.lastPeriod.duration ? cache.lastPeriod.balanceSum / cache.lastPeriod.duration : accountInformation.balance;
                                    cache.record.minEquity = Math.min(cache.record.minEquity, price.equity);
                                    cache.record.maxEquity = Math.max(cache.record.maxEquity, price.equity);
                                    cache.record.lastEquity = equity;
                                    cache.record.minBalance = Math.min(cache.record.minBalance, accountInformation.balance);
                                    cache.record.maxBalance = Math.max(cache.record.maxBalance, accountInformation.balance);
                                    cache.record.lastBalance = accountInformation.balance;
                                    /**
             * due to calculation inaccuracy, averageEquity will never match the previous value
             * therefore, floor before comparing
             */ if (cache.lastPeriod.startBrokerTime) {
                                        const newInfo = {
                                            startBrokerTime: cache.lastPeriod.startBrokerTime,
                                            endBrokerTime: cache.lastPeriod.endBrokerTime,
                                            averageBalance: cache.record.averageBalance,
                                            minBalance: cache.record.minBalance,
                                            maxBalance: cache.record.maxBalance,
                                            averageEquity: Math.floor(cache.record.averageEquity),
                                            minEquity: cache.record.minEquity,
                                            maxEquity: cache.record.maxEquity,
                                            lastBalance: cache.record.lastBalance,
                                            lastEquity: cache.record.lastEquity
                                        };
                                        if (JSON.stringify(previousInfo) !== JSON.stringify(newInfo)) {
                                            Object.values(getAccountListeners()).forEach((accountListener)=>{
                                                accountListener.onEquityRecordUpdated([
                                                    newInfo
                                                ]);
                                            });
                                        }
                                    }
                                }
                            }
                        } catch (err) {
                            Object.values(getAccountListeners()).forEach((accountListener)=>{
                                accountListener.onError(err);
                            });
                            _this._logger.error('Error processing onSymbolPriceUpdated event for ' + `equity chart listener for account ${accountId}`, err);
                        }
                    })();
                }
                onAccountInformationUpdated(instanceIndex, accountInformation) {
                    var _this = this;
                    return _async_to_generator(function*() {
                        try {
                            const balance = accountInformation.balance;
                            cache.lastPeriod.balance = balance;
                            cache.lastPeriod.lastBalance = balance;
                            cache.record.lastBalance = balance;
                            cache.record.minBalance = Math.min(cache.record.minBalance, balance);
                            cache.record.maxBalance = Math.max(cache.record.minBalance, balance);
                        } catch (err) {
                            Object.values(getAccountListeners()).forEach((accountListener)=>{
                                accountListener.onError(err);
                            });
                            _this._logger.error('Error processing onAccountInformationUpdated event for ' + `equity chart listener for account ${accountId}`, err);
                        }
                    })();
                }
            };
            const listenerId = randomstring.generate(10);
            const accountListeners = _this.getAccountListeners(accountId);
            accountListeners[listenerId] = listener;
            _this._accountsByListenerId[listenerId] = accountId;
            const account = yield _this._metaApi.metatraderAccountApi.getAccount(accountId);
            let isDeployed = false;
            while(!isDeployed){
                try {
                    yield account.waitDeployed();
                    isDeployed = true;
                } catch (err) {
                    listener.onError(err);
                    _this._logger.error(`Error wait for account ${accountId} to deploy, retrying`, err);
                    yield new Promise((res)=>setTimeout(res, retryIntervalInSeconds * 1000));
                    retryIntervalInSeconds = Math.min(retryIntervalInSeconds * 2, 300);
                }
            }
            if (!_this._equityChartConnections[accountId]) {
                retryIntervalInSeconds = _this._retryIntervalInSeconds;
                connection = account.getStreamingConnection();
                _this._equityChartConnections[accountId] = connection;
                const syncListener = new EquityChartStreamListener();
                connection.addSynchronizationListener(syncListener);
                let isSynchronized = false;
                while(!isSynchronized){
                    try {
                        yield connection.connect();
                        yield connection.waitSynchronized();
                        isSynchronized = true;
                    } catch (err) {
                        listener.onError(err);
                        _this._logger.error(`Error configuring equity chart stream listener for account ${accountId}, retrying`, err);
                        yield new Promise((res)=>setTimeout(res, retryIntervalInSeconds * 1000));
                        retryIntervalInSeconds = Math.min(retryIntervalInSeconds * 2, 300);
                    }
                }
                retryIntervalInSeconds = _this._retryIntervalInSeconds;
            } else {
                connection = _this._equityChartConnections[accountId];
                if (!connection.healthMonitor.healthStatus.synchronized) {
                    if (!_this._pendingInitalizationResolves[accountId]) {
                        _this._pendingInitalizationResolves[accountId] = [];
                    }
                    let resolveInitialize;
                    let initializePromise = new Promise((res, rej)=>{
                        resolveInitialize = res;
                    });
                    _this._pendingInitalizationResolves[accountId].push(resolveInitialize);
                    yield initializePromise;
                }
            }
            let initialData = [];
            while(!initialData.length){
                try {
                    initialData = yield equityTrackingClient.getEquityChart(accountId, startTime, undefined, true);
                    if (initialData.length) {
                        const lastItem = initialData.slice(-1)[0];
                        listener.onEquityRecordUpdated(initialData);
                        cache.lastPeriod = {
                            duration: lastItem.duration,
                            equitySum: lastItem.equitySum,
                            balanceSum: lastItem.balanceSum,
                            startBrokerTime: lastItem.startBrokerTime,
                            endBrokerTime: lastItem.endBrokerTime,
                            brokerTime: lastItem.brokerTime,
                            averageEquity: Math.floor(lastItem.averageEquity),
                            minEquity: lastItem.minEquity,
                            maxEquity: lastItem.maxEquity,
                            averageBalance: lastItem.averageBalance,
                            minBalance: lastItem.minBalance,
                            maxBalance: lastItem.maxBalance,
                            lastBalance: lastItem.lastBalance,
                            lastEquity: lastItem.lastEquity
                        };
                        cache.record = cache.lastPeriod;
                    }
                } catch (err) {
                    listener.onError(err);
                    _this._logger.error(`Failed initialize equity chart data for account ${accountId}`, err);
                    yield new Promise((res)=>setTimeout(res, retryIntervalInSeconds * 1000));
                    retryIntervalInSeconds = Math.min(retryIntervalInSeconds * 2, 300);
                }
            }
            return listenerId;
        })();
    }
    /**
   * Removes equity chart event listener by id
   * @param {String} listenerId listener id
   */ removeEquityChartListener(listenerId) {
        if (this._accountsByListenerId[listenerId]) {
            const accountId = this._accountsByListenerId[listenerId];
            delete this._accountSynchronizationFlags[accountId];
            delete this._accountsByListenerId[listenerId];
            if (this._equityChartListeners[accountId]) {
                delete this._equityChartListeners[accountId][listenerId];
            }
            if (this._equityChartConnections[accountId] && !Object.keys(this._equityChartListeners[accountId]).length) {
                this._equityChartConnections[accountId].close();
                delete this._equityChartConnections[accountId];
            }
        }
    }
    /**
   * Constructs equity chart event listener manager instance
   * @param {DomainClient} domainClient domain client
   * @param {EquityTrackingClient} equityTrackingClient equity tracking client
   * @param {MetaApi} metaApi metaApi SDK instance
   */ constructor(domainClient, equityTrackingClient, metaApi){
        this._domainClient = domainClient;
        this._equityTrackingClient = equityTrackingClient;
        this._metaApi = metaApi;
        this._equityChartListeners = {};
        this._accountsByListenerId = {};
        this._equityChartConnections = {};
        this._equityChartCaches = {};
        this._accountSynchronizationFlags = {};
        this._pendingInitalizationResolves = {};
        this._retryIntervalInSeconds = 1;
        this._logger = LoggerManager.getLogger('EquityChartStreamManager');
    }
};
/**
 * Manager for handling equity chart event listeners
 */ export { EquityChartStreamManager as default };

//# sourceMappingURL=data:application/json;base64,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