'use strict';
function asyncGeneratorStep(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
import HistoryStorage from './historyStorage';
import HistoryDatabase from './historyDatabase/index';
import { AVLTree } from 'binary-search-tree';
import LoggerManager from '../logger';
let MemoryHistoryStorage = class MemoryHistoryStorage extends HistoryStorage {
    /**
   * Initializes the storage and loads required data from a persistent storage
   * @param {string} accountId account id
   * @param {string} [application] application. Default is MetaApi
   */ initialize(accountId, application = 'MetaApi') {
        var _this = this, _superprop_get_initialize = ()=>super.initialize;
        return _async_to_generator(function*() {
            yield _superprop_get_initialize().call(_this, accountId, application);
            let { deals, historyOrders } = yield _this._historyDatabase.loadHistory(accountId, application);
            for (let deal of deals){
                yield _this._addDeal(deal, true);
            }
            for (let historyOrder of historyOrders){
                yield _this._addHistoryOrder(historyOrder, true);
            }
        })();
    }
    /**
   * Resets the storage. Intended for use in tests
   * @returns {Promise} promise when the history is removed
   */ clear() {
        var _this = this;
        return _async_to_generator(function*() {
            _this._reset();
            yield _this._historyDatabase.clear(_this._accountId, _this._application);
        })();
    }
    /**
   * Returns the time of the last history order record stored in the history storage
   * @param {Number} [instanceNumber] index of an account instance connected
   * @returns {Date} the time of the last history order record stored in the history storage
   */ lastHistoryOrderTime(instanceNumber) {
        return this._maxHistoryOrderTime;
    }
    /**
   * Returns the time of the last history deal record stored in the history storage
   * @param {Number} [instanceNumber] index of an account instance connected
   * @returns {Date} the time of the last history deal record stored in the history storage
   */ lastDealTime(instanceNumber) {
        return this._maxDealTime;
    }
    /**
   * Invoked when a new MetaTrader history order is added
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderOrder} historyOrder new MetaTrader history order
   */ onHistoryOrderAdded(instanceIndex, historyOrder) {
        var _this = this;
        return _async_to_generator(function*() {
            yield _this._addHistoryOrder(historyOrder);
        })();
    }
    /**
   * Invoked when a new MetaTrader history deal is added
   * @param {String} instanceIndex index of an account instance connected
   * @param {MetatraderDeal} deal new MetaTrader history deal
   */ onDealAdded(instanceIndex, deal) {
        var _this = this;
        return _async_to_generator(function*() {
            yield _this._addDeal(deal);
        })();
    }
    /**
   * Returns all deals
   * @returns {Array<MetatraderDeal>} all deals
   */ get deals() {
        return this.getDealsByTimeRange(new Date(0), new Date(8640000000000000));
    }
    /**
   * Returns deals by ticket id
   * @param {string} id ticket id
   * @returns {Array<MetatraderDeal>} deals found
   */ getDealsByTicket(id) {
        let deals = Object.values(this._dealsByTicket[id] || {});
        deals.sort(this._dealsComparator);
        return deals;
    }
    /**
   * Returns deals by position id
   * @param {string} positionId position id
   * @returns {Array<MetatraderDeal>} deals found
   */ getDealsByPosition(positionId) {
        let deals = Object.values(this._dealsByPosition[positionId] || {});
        deals.sort(this._dealsComparator);
        return deals;
    }
    /**
   * Returns deals by time range
   * @param startTime start time, inclusive
   * @param endTime end time, inclusive
   * @returns {Array<MetatraderDeal>} deals found
   */ getDealsByTimeRange(startTime, endTime) {
        let deals = this._dealsByTime.betweenBounds({
            $gte: {
                time: startTime,
                id: 0,
                entryType: ''
            },
            $lte: {
                time: endTime,
                id: Number.MAX_VALUE,
                entryType: ''
            }
        });
        return deals;
    }
    /**
   * Returns all history orders
   * @returns {Array<MetatraderOrder>} all history orders
   */ get historyOrders() {
        return this.getHistoryOrdersByTimeRange(new Date(0), new Date(8640000000000000));
    }
    /**
   * Returns history orders by ticket id
   * @param {string} id ticket id
   * @returns {Array<MetatraderOrder>} history orders found
   */ getHistoryOrdersByTicket(id) {
        let historyOrders = Object.values(this._historyOrdersByTicket[id] || {});
        historyOrders.sort(this._historyOrdersComparator);
        return historyOrders;
    }
    /**
   * Returns history orders by position id
   * @param {string} positionId position id
   * @returns {Array<MetatraderOrder>} history orders found
   */ getHistoryOrdersByPosition(positionId) {
        let historyOrders = Object.values(this._historyOrdersByPosition[positionId] || {});
        historyOrders.sort(this._historyOrdersComparator);
        return historyOrders;
    }
    /**
   * Returns history orders by time range
   * @param startTime start time, inclusive
   * @param endTime end time, inclusive
   * @returns {Array<MetatraderOrder>} hisotry orders found
   */ getHistoryOrdersByTimeRange(startTime, endTime) {
        let historyOrders = this._historyOrdersByTime.betweenBounds({
            $gte: {
                doneTime: startTime,
                id: 0,
                type: '',
                state: ''
            },
            $lte: {
                doneTime: endTime,
                id: Number.MAX_VALUE,
                type: '',
                state: ''
            }
        });
        return historyOrders;
    }
    /**
   * Invoked when a synchronization of history deals on a MetaTrader account have finished to indicate progress of an
   * initial terminal state synchronization
   * @param {String} instanceIndex index of an account instance connected
   * @param {String} synchronizationId synchronization request id
   * @return {Promise} promise which resolves when the asynchronous event is processed
   */ onDealsSynchronized(instanceIndex, synchronizationId) {
        var _this = this, _superprop_get_onDealsSynchronized = ()=>super.onDealsSynchronized;
        return _async_to_generator(function*() {
            yield _this._flushDatabase();
            yield _superprop_get_onDealsSynchronized().call(_this, instanceIndex, synchronizationId);
        })();
    }
    _reset() {
        this._orderSynchronizationFinished = {};
        this._dealSynchronizationFinished = {};
        this._dealsByTicket = {};
        this._dealsByPosition = {};
        this._historyOrdersByTicket = {};
        this._historyOrdersByPosition = {};
        // eslint-disable-next-line complexity
        this._historyOrdersComparator = (o1, o2)=>{
            let timeDiff = (o1.doneTime || new Date(0)).getTime() - (o2.doneTime || new Date(0)).getTime();
            if (timeDiff === 0) {
                let idDiff = o1.id - o2.id;
                if (idDiff === 0) {
                    if (o1.type > o2.type) {
                        return 1;
                    } else if (o1.type < o2.type) {
                        return -1;
                    } else {
                        if (o1.state > o2.state) {
                            return 1;
                        } else if (o1.state < o2.state) {
                            return -1;
                        } else {
                            return 0;
                        }
                    }
                } else {
                    return idDiff;
                }
            } else {
                return timeDiff;
            }
        };
        this._historyOrdersByTime = new AVLTree({
            compareKeys: this._historyOrdersComparator
        });
        this._dealsComparator = (d1, d2)=>{
            let timeDiff = (d1.time || new Date(0)).getTime() - (d2.time || new Date(0)).getTime();
            if (timeDiff === 0) {
                let idDiff = d1.id - d2.id;
                if (idDiff === 0) {
                    if (d1.entryType > d2.entryType) {
                        return 1;
                    } else if (d1.entryType < d2.entryType) {
                        return -1;
                    } else {
                        return 0;
                    }
                } else {
                    return idDiff;
                }
            } else {
                return timeDiff;
            }
        };
        this._dealsByTime = new AVLTree({
            compareKeys: this._dealsComparator
        });
        this._maxHistoryOrderTime = new Date(0);
        this._maxDealTime = new Date(0);
        this._newHistoryOrders = [];
        this._newDeals = [];
        clearTimeout(this._flushTimeout);
        delete this._flushTimeout;
    }
    // eslint-disable-next-line complexity
    _addDeal(deal, existing) {
        var _this = this;
        return _async_to_generator(function*() {
            let key = _this._getDealKey(deal);
            _this._dealsByTicket[deal.id] = _this._dealsByTicket[deal.id] || {};
            let newDeal = !existing && !_this._dealsByTicket[deal.id][key];
            _this._dealsByTicket[deal.id][key] = deal;
            if (deal.positionId) {
                _this._dealsByPosition[deal.positionId] = _this._dealsByPosition[deal.positionId] || {};
                _this._dealsByPosition[deal.positionId][key] = deal;
            }
            _this._dealsByTime.delete(deal);
            _this._dealsByTime.insert(deal, deal);
            if (deal.time && (!_this._maxDealTime || _this._maxDealTime.getTime() < deal.time.getTime())) {
                _this._maxDealTime = deal.time;
            }
            if (newDeal) {
                _this._newDeals.push(deal);
                clearTimeout(_this._flushTimeout);
                _this._flushTimeout = setTimeout(_this._flushDatabase.bind(_this), 5000);
            }
        })();
    }
    _getDealKey(deal) {
        return (deal.time || new Date(0)).toISOString() + ':' + deal.id + ':' + deal.entryType;
    }
    // eslint-disable-next-line complexity
    _addHistoryOrder(historyOrder, existing) {
        var _this = this;
        return _async_to_generator(function*() {
            let key = _this._getHistoryOrderKey(historyOrder);
            _this._historyOrdersByTicket[historyOrder.id] = _this._historyOrdersByTicket[historyOrder.id] || {};
            let newHistoryOrder = !existing && !_this._historyOrdersByTicket[historyOrder.id][key];
            _this._historyOrdersByTicket[historyOrder.id][key] = historyOrder;
            if (historyOrder.positionId) {
                _this._historyOrdersByPosition[historyOrder.positionId] = _this._historyOrdersByPosition[historyOrder.positionId] || {};
                _this._historyOrdersByPosition[historyOrder.positionId][key] = historyOrder;
            }
            _this._historyOrdersByTime.delete(historyOrder);
            _this._historyOrdersByTime.insert(historyOrder, historyOrder);
            if (historyOrder.doneTime && (!_this._maxHistoryOrderTime || _this._maxHistoryOrderTime.getTime() < historyOrder.doneTime.getTime())) {
                _this._maxHistoryOrderTime = historyOrder.doneTime;
            }
            if (newHistoryOrder) {
                _this._newHistoryOrders.push(historyOrder);
                clearTimeout(_this._flushTimeout);
                _this._flushTimeout = setTimeout(_this._flushDatabase.bind(_this), 5000);
            }
        })();
    }
    _getHistoryOrderKey(historyOrder) {
        return (historyOrder.doneTime || new Date(0)).toISOString() + ':' + historyOrder.id + ':' + historyOrder.type + ':' + historyOrder.status;
    }
    _flushDatabase() {
        var _this = this;
        return _async_to_generator(function*() {
            if (_this._flushPromise) {
                yield _this._flushPromise;
            }
            if (_this._flushRunning) {
                return;
            }
            _this._flushRunning = true;
            let resolve;
            _this._flushPromise = new Promise((res)=>resolve = res);
            try {
                yield _this._historyDatabase.flush(_this._accountId, _this._application, _this._newHistoryOrders, _this._newDeals);
                _this._newHistoryOrders = [];
                _this._newDeals = [];
                _this._logger.debug(`${_this._accountId}: flushed history db`);
            } catch (err) {
                _this._logger.warn(`${_this._accountId}: error flushing history db`, err);
                _this._flushTimeout = setTimeout(_this._flushDatabase.bind(_this), 15000);
            } finally{
                resolve();
                _this._flushRunning = false;
            }
        })();
    }
    /**
   * Constructs the in-memory history store instance
   */ constructor(){
        super();
        this._historyDatabase = HistoryDatabase.getInstance();
        this._reset();
        this._logger = LoggerManager.getLogger('MemoryHistoryStorage');
    }
};
/**
 * History storage which stores MetaTrader history in RAM
 */ export { MemoryHistoryStorage as default };

//# sourceMappingURL=data:application/json;base64,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