'use strict';
function asyncGeneratorStep(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
import socketIO from 'socket.io-client';
import LoggerManager from '../../logger';
let LatencyService = class LatencyService {
    /**
   * Stops the service
   */ stop() {
        clearInterval(this._refreshRegionLatencyInterval);
    }
    /**
   * Returns the list of regions sorted by latency
   * @returns {String[]} list of regions sorted by latency
   */ get regionsSortedByLatency() {
        const regions = Object.keys(this._latencyCache);
        regions.sort((a, b)=>this._latencyCache[a] - this._latencyCache[b]);
        return regions;
    }
    /**
   * Invoked when an instance has been disconnected
   * @param {String} instanceId instance id
   */ onDisconnected(instanceId) {
        try {
            const accountId = this._getAccountIdFromInstance(instanceId);
            const disconnectedRegion = this._getRegionFromInstance(instanceId);
            this._disconnectInstance(instanceId);
            const instances = this._getAccountInstances(accountId);
            if (!instances.map((instance)=>this._connectedInstancesCache[instance]).includes(true)) {
                const regions = this._getAccountRegions(accountId);
                regions.filter((region)=>region !== disconnectedRegion).forEach((region)=>this._subscribeAccountReplica(accountId, region));
            }
        } catch (err) {
            this._logger.error(`Failed to process onDisconnected event for instance ${instanceId}`, err);
        }
    }
    /**
   * Invoked when an account has been unsubscribed
   * @param {String} accountId account id
   */ onUnsubscribe(accountId) {
        try {
            const region = this._websocketClient.getAccountRegion(accountId);
            const primaryAccountId = this._websocketClient.accountsByReplicaId[accountId];
            const instances = this._getAccountInstances(primaryAccountId);
            instances.filter((instanceId)=>instanceId.startsWith(`${primaryAccountId}:${region}:`)).forEach((instanceId)=>this._disconnectInstance(instanceId));
        } catch (err) {
            this._logger.error(`Failed to process onUnsubscribe event for account ${accountId}`, err);
        }
    }
    /**
   * Invoked when an instance has been connected
   * @param {String} instanceId instance id
   */ onConnected(instanceId) {
        var _this = this;
        return _async_to_generator(function*() {
            try {
                _this._connectedInstancesCache[instanceId] = true;
                const accountId = _this._getAccountIdFromInstance(instanceId);
                const region = _this._getRegionFromInstance(instanceId);
                if (!_this._latencyCache[region]) {
                    yield _this._refreshLatency(region);
                }
                const instances = _this.getActiveAccountInstances(accountId);
                const synchronizedInstances = _this.getSynchronizedAccountInstances(accountId);
                const regions = instances.map((instance)=>_this._getRegionFromInstance(instance));
                if (instances.length > 1 && !synchronizedInstances.length) {
                    const regionsToDisconnect = _this.regionsSortedByLatency.filter((sortedRegion)=>regions.includes(sortedRegion)).slice(1);
                    regionsToDisconnect.forEach((regionItem)=>{
                        _this._websocketClient.unsubscribe(_this._websocketClient.accountReplicas[accountId][regionItem]);
                        _this._websocketClient.unsubscribeAccountRegion(accountId, regionItem);
                    });
                }
                if (_this._waitConnectPromises[accountId]) {
                    _this._waitConnectPromises[accountId].resolve();
                    delete _this._waitConnectPromises[accountId];
                }
            } catch (err) {
                _this._logger.error(`Failed to process onConnected event for instance ${instanceId}`, err);
            }
        })();
    }
    /**
   * Invoked when an instance has been synchronized
   * @param {String} instanceId instance id
   */ onDealsSynchronized(instanceId) {
        var _this = this;
        return _async_to_generator(function*() {
            try {
                _this._synchronizedInstancesCache[instanceId] = true;
                const accountId = _this._getAccountIdFromInstance(instanceId);
                const region = _this._getRegionFromInstance(instanceId);
                if (!_this._latencyCache[region]) {
                    yield _this._refreshLatency(region);
                }
                const instances = _this.getSynchronizedAccountInstances(accountId);
                const regions = [
                    ...new Set(instances.map((instance)=>_this._getRegionFromInstance(instance)))
                ];
                if (instances.length > 1) {
                    const regionsToDisconnect = _this.regionsSortedByLatency.filter((sortedRegion)=>regions.includes(sortedRegion)).slice(1);
                    regionsToDisconnect.forEach((regionItem)=>{
                        _this._websocketClient.unsubscribe(_this._websocketClient.accountReplicas[accountId][regionItem]);
                        _this._websocketClient.unsubscribeAccountRegion(accountId, regionItem);
                    });
                }
            } catch (err) {
                _this._logger.error(`Failed to process onDealsSynchronized event for instance ${instanceId}`, err);
            }
        })();
    }
    /**
   * Returns the list of currently connected account instances
   * @param {String} accountId account id
   * @returns {String[]} list of connected account instances
   */ getActiveAccountInstances(accountId) {
        return this._getAccountInstances(accountId).filter((instance)=>this._connectedInstancesCache[instance]);
    }
    /**
   * Returns the list of currently synchronized account instances
   * @param {String} accountId account id
   * @returns {String[]} list of synchronized account instances
   */ getSynchronizedAccountInstances(accountId) {
        return this._getAccountInstances(accountId).filter((instance)=>this._synchronizedInstancesCache[instance]);
    }
    /**
   * Waits for connected instance
   * @param {String} accountId account id 
   * @returns {String} instance id
   */ waitConnectedInstance(accountId) {
        var _this = this;
        return _async_to_generator(function*() {
            let instances = _this.getActiveAccountInstances(accountId);
            if (!instances.length) {
                if (!_this._waitConnectPromises[accountId]) {
                    let resolve;
                    let promise = new Promise((res, rej)=>{
                        resolve = res;
                    });
                    _this._waitConnectPromises[accountId] = {
                        promise,
                        resolve
                    };
                }
                yield _this._waitConnectPromises[accountId].promise;
                instances = _this.getActiveAccountInstances(accountId);
            }
            return instances[0];
        })();
    }
    _getAccountInstances(accountId) {
        return Object.keys(this._connectedInstancesCache).filter((instanceId)=>instanceId.startsWith(`${accountId}:`));
    }
    _getAccountRegions(accountId) {
        const regions = [];
        const instances = this._getAccountInstances(accountId);
        instances.forEach((instance)=>{
            const region = this._getRegionFromInstance(instance);
            if (!regions.includes(region)) {
                regions.push(region);
            }
        });
        return regions;
    }
    _getAccountIdFromInstance(instanceId) {
        return instanceId.split(':')[0];
    }
    _getRegionFromInstance(instanceId) {
        return instanceId.split(':')[1];
    }
    _disconnectInstance(instanceId) {
        this._connectedInstancesCache[instanceId] = false;
        if (this._synchronizedInstancesCache[instanceId]) {
            this._synchronizedInstancesCache[instanceId] = false;
        }
    }
    _subscribeAccountReplica(accountId, region) {
        const instanceId = this._websocketClient.accountReplicas[accountId][region];
        if (instanceId) {
            this._websocketClient.ensureSubscribe(instanceId, 0);
            this._websocketClient.ensureSubscribe(instanceId, 1);
        }
    }
    _refreshRegionLatencyJob() {
        var _this = this;
        return _async_to_generator(function*() {
            for (let region of Object.keys(_this._latencyCache)){
                yield _this._refreshLatency(region);
            }
            // For every account, switch to a better region if such exists
            const accountIds = [];
            Object.keys(_this._connectedInstancesCache).filter((instanceId)=>_this._connectedInstancesCache[instanceId]).forEach((instanceId)=>{
                const accountId = _this._getAccountIdFromInstance(instanceId);
                if (!accountIds.includes(accountId)) {
                    accountIds.push(accountId);
                }
            });
            const sortedRegions = _this.regionsSortedByLatency;
            accountIds.forEach((accountId)=>{
                const accountRegions = _this._getAccountRegions(accountId);
                const activeInstances = _this.getActiveAccountInstances(accountId);
                if (activeInstances.length === 1) {
                    const activeInstance = activeInstances[0];
                    const activeRegion = _this._getRegionFromInstance(activeInstance);
                    const accountBestRegions = sortedRegions.filter((region)=>accountRegions.includes(region));
                    if (accountBestRegions[0] !== activeRegion) {
                        _this._subscribeAccountReplica(accountId, accountBestRegions[0]);
                    }
                }
            });
        })();
    }
    _refreshLatency(region) {
        var _this = this;
        return _async_to_generator(function*() {
            if (_this._refreshPromisesByRegion[region]) {
                return yield _this._refreshPromisesByRegion[region];
            }
            let resolve;
            _this._refreshPromisesByRegion[region] = new Promise((res, rej)=>{
                resolve = res;
            });
            const serverUrl = yield _this._websocketClient.getUrlSettings(0, region);
            const startDate = Date.now();
            const socketInstance = socketIO(serverUrl.url, {
                path: '/ws',
                reconnection: true,
                reconnectionDelay: 1000,
                reconnectionDelayMax: 5000,
                reconnectionAttempts: Infinity,
                timeout: _this._connectTimeout,
                query: {
                    'auth-token': _this._token,
                    protocol: 3
                }
            });
            socketInstance.on('connect', /*#__PURE__*/ _async_to_generator(function*() {
                resolve();
                const latency = Date.now() - startDate;
                _this._latencyCache[region] = latency;
                socketInstance.close();
            }));
            yield _this._refreshPromisesByRegion[region];
            delete _this._refreshPromisesByRegion[region];
        })();
    }
    /**
   * Constructs latency service instance
   * @param {MetaApiWebsocketClient} websocketClient MetaApi websocket client
   * @param {String} token authorization token
   * @param {Number} connectTimeout websocket connect timeout in seconds
   */ constructor(websocketClient, token, connectTimeout){
        this._websocketClient = websocketClient;
        this._token = token;
        this._connectTimeout = connectTimeout;
        this._latencyCache = {};
        this._connectedInstancesCache = {};
        this._synchronizedInstancesCache = {};
        this._refreshPromisesByRegion = {};
        this._waitConnectPromises = {};
        this._logger = LoggerManager.getLogger('LatencyService');
        this._refreshRegionLatencyJob = this._refreshRegionLatencyJob.bind(this);
        this._refreshRegionLatencyInterval = setInterval(this._refreshRegionLatencyJob, 15 * 60 * 1000);
    }
};
/**
 * Service for managing account replicas based on region latency
 */ export { LatencyService as default };

//# sourceMappingURL=data:application/json;base64,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