import randomstring from 'randomstring';

/**
 * CopyFactory client for signal requests
 */
export default class SubscriberSignalClient {

  /**
   * Constructs CopyFactory signal client instance
   * @param {string} accountId account id
   * @param {Object} host host data
   * @param {DomainClient} domainClient domain client
   */
  constructor(accountId, host, domainClient) {
    this._accountId = accountId;
    this._domainClient = domainClient;
    this._host = host;
  }

  /**
   * CopyFactory trading signal
   * @typedef {Object} CopyFactoryTradingSignal
   * @property {CopyFactoryStrategyIdAndName} strategy strategy the signal arrived from
   * @property {String} positionId id of the position the signal was generated from
   * @property {Date} time signal time
   * @property {String} symbol symbol traded
   * @property {String} type type of the trade (one of market, limit, stop)
   * @property {String} side side of the trade (one of buy, sell, close)
   * @property {Number} [openPrice] open price for limit and stop orders
   * @property {Number} [stopLoss] stop loss price
   * @property {Number} [takeProfit] take profit price
   * @property {Number} signalVolume the signal volume
   * @property {Number} subscriberVolume the volume already open on subscriber side
   * @property {Number} subscriberProfit total profit of the position on subscriber side
   * @property {Date} closeAfter the time the signal will be automatically closed at
   * @property {Boolean} [closeOnly] flag indicating that only closing side of this signal will be copied
   */

  /**
   * Returns trading signals the subscriber is subscribed to. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/getTradingSignals/
   * @returns {Promise<Array<CopyFactoryTradingSignal>>}
   */
  getTradingSignals() {
    const opts = {
      url: `/users/current/subscribers/${this._accountId}/signals`,
      method: 'GET',
      headers: {
        'auth-token': this._domainClient.token
      },
      json: true
    };
    return this._domainClient.requestSignal(opts, this._host, this._accountId);
  }

}